<?php
/**
 * Admin Settings for Tenant Referral Program
 * 
 * Handles the WordPress admin settings page
 */

if (!defined('ABSPATH')) {
    exit;
}

class Tenant_Referral_Admin_Settings {
    
    /**
     * Settings page slug
     */
    private $page_slug = 'tenant-referral-program';
    
    /**
     * Settings group
     */
    private $settings_group = 'trp_settings';
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'init_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('wp_ajax_trp_test_api_key', array($this, 'ajax_test_api_key'));
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_options_page(
            __('Tenant Referral Program', 'tenant-referral-program'),
            __('Referral Program', 'tenant-referral-program'),
            'manage_options',
            $this->page_slug,
            array($this, 'render_settings_page')
        );
    }
    
    /**
     * Initialize settings
     */
    public function init_settings() {
        // Register settings
        register_setting(
            $this->settings_group,
            'trp_api_key',
            array(
                'type' => 'string',
                'sanitize_callback' => array($this, 'sanitize_api_key'),
                'show_in_rest' => false // Keep secure
            )
        );
        
        register_setting(
            $this->settings_group,
            'trp_base_url',
            array(
                'type' => 'string',
                'sanitize_callback' => array($this, 'sanitize_url'),
                'default' => 'https://offertown.net'
            )
        );
        
        register_setting(
            $this->settings_group,
            'trp_cookie_expiry_days',
            array(
                'type' => 'integer',
                'sanitize_callback' => 'absint',
                'default' => TRP_COOKIE_EXPIRY_DAYS
            )
        );
        
        register_setting(
            $this->settings_group,
            'trp_cookie_name',
            array(
                'type' => 'string',
                'sanitize_callback' => array($this, 'sanitize_cookie_name'),
                'default' => TRP_COOKIE_NAME
            )
        );

        register_setting(
            $this->settings_group,
            'trp_agreement_text',
            array(
                'type' => 'string',
                'sanitize_callback' => array($this, 'sanitize_agreement_text'),
                'default' => 'I agree to the Terms & Conditions'
            )
        );

        register_setting(
            $this->settings_group,
            'trp_terms_url',
            array(
                'type' => 'string',
                'sanitize_callback' => array($this, 'sanitize_terms_url'),
                'default' => '/terms-and-conditions/'
            )
        );

        // Add settings sections
        add_settings_section(
            'trp_general_section',
            __('General Settings', 'tenant-referral-program'),
            array($this, 'render_general_section'),
            $this->page_slug
        );
        
        add_settings_section(
            'trp_tracking_section',
            __('Tracking Settings', 'tenant-referral-program'),
            array($this, 'render_tracking_section'),
            $this->page_slug
        );
        
        // Add settings fields
        add_settings_field(
            'trp_api_key',
            __('API Key', 'tenant-referral-program'),
            array($this, 'render_api_key_field'),
            $this->page_slug,
            'trp_general_section'
        );
        
        add_settings_field(
            'trp_base_url',
            __('Referral Program Base URL', 'tenant-referral-program'),
            array($this, 'render_base_url_field'),
            $this->page_slug,
            'trp_general_section'
        );
        
        add_settings_field(
            'trp_cookie_expiry_days',
            __('Cookie Expiration (Days)', 'tenant-referral-program'),
            array($this, 'render_cookie_expiry_field'),
            $this->page_slug,
            'trp_tracking_section'
        );
        
        add_settings_field(
            'trp_cookie_name',
            __('Cookie Name', 'tenant-referral-program'),
            array($this, 'render_cookie_name_field'),
            $this->page_slug,
            'trp_tracking_section'
        );

        add_settings_field(
            'trp_agreement_text',
            __('Agreement Text', 'tenant-referral-program'),
            array($this, 'render_agreement_text_field'),
            $this->page_slug,
            'trp_general_section'
        );

        add_settings_field(
            'trp_terms_url',
            __('Terms & Conditions URL', 'tenant-referral-program'),
            array($this, 'render_terms_url_field'),
            $this->page_slug,
            'trp_general_section'
        );

        // Add debug/testing fields
        add_settings_field(
            'trp_bypass_api_validation',
            __('Bypass API Validation', 'tenant-referral-program'),
            array($this, 'render_bypass_validation_field'),
            $this->page_slug,
            'trp_general_section'
        );
        
        add_settings_field(
            'trp_development_mode',
            __('Development Mode', 'tenant-referral-program'),
            array($this, 'render_development_mode_field'),
            $this->page_slug,
            'trp_general_section'
        );
        
        // Register the bypass setting
        register_setting(
            $this->settings_group,
            'trp_bypass_api_validation',
            array(
                'type' => 'boolean',
                'sanitize_callback' => 'rest_sanitize_boolean',
                'default' => false
            )
        );
        
        // Register the development mode setting
        register_setting(
            $this->settings_group,
            'trp_development_mode',
            array(
                'type' => 'boolean',
                'sanitize_callback' => 'rest_sanitize_boolean',
                'default' => false
            )
        );
    }
    
    /**
     * Sanitize URL input and enforce HTTPS
     */
    public function sanitize_url($input) {
        $url = esc_url_raw($input);
        $clean_url = rtrim($url, '/');
        
        // Enforce HTTPS for security
        if (!empty($clean_url)) {
            $parsed = parse_url($clean_url);
            
            // Check if it's a valid URL
            if ($parsed === false || !isset($parsed['scheme']) || !isset($parsed['host'])) {
                add_settings_error('trp_base_url', 'invalid_url', 
                    __('Please enter a valid URL with protocol (https://)', 'tenant-referral-program'));
                return get_option('trp_base_url', 'https://offertown.net');
            }
            
            // Enforce HTTPS (preserve port); allow dev hosts during WP_DEBUG
            if ($parsed['scheme'] !== 'https') {
                $host = isset($parsed['host']) ? $parsed['host'] : '';
                $is_dev_host = in_array($host, array('localhost', '127.0.0.1', '::1'))
                    || strpos($host, '.local') !== false
                    || strpos($host, 'ngrok') !== false;

                if ($is_dev_host && defined('WP_DEBUG') && WP_DEBUG) {
                    // Do not force HTTPS for dev hosts in debug mode
                    add_settings_error('trp_base_url', 'https_dev_skip',
                        __('Development host detected in debug mode; HTTPS not forced. Consider using HTTPS when possible.', 'tenant-referral-program'),
                        'notice-warning');
                } else {
                    // Convert to HTTPS and preserve port + path + query + fragment
                    $port     = isset($parsed['port']) ? ':' . $parsed['port'] : '';
                    $path     = isset($parsed['path']) ? $parsed['path'] : '';
                    $query    = isset($parsed['query']) ? '?' . $parsed['query'] : '';
                    $fragment = isset($parsed['fragment']) ? '#' . $parsed['fragment'] : '';
                    $clean_url = 'https://' . $parsed['host'] . $port . $path . $query . $fragment;
                    add_settings_error('trp_base_url', 'https_enforced',
                        __('URL has been converted to HTTPS for security.', 'tenant-referral-program'),
                        'notice-info');
                }
            }
            
            // Warn about localhost/development URLs
            if (in_array($parsed['host'], array('localhost', '127.0.0.1', '::1')) || 
                strpos($parsed['host'], '.local') !== false ||
                strpos($parsed['host'], 'ngrok') !== false) {
                add_settings_error('trp_base_url', 'dev_url_warning', 
                    __('Warning: This appears to be a development URL. Make sure to update it for production.', 'tenant-referral-program'), 'notice-warning');
            }
        }
        
        return $clean_url;
    }
    
    /**
     * Sanitize cookie name input
     */
    public function sanitize_cookie_name($input) {
        $cookie_name = sanitize_text_field($input);
        
        // If empty, return default
        if (empty($cookie_name)) {
            return TRP_COOKIE_NAME;
        }
        
        // Remove any characters that are not safe for cookie names
        // Allow letters, numbers, underscores, dashes, and dots
        $cookie_name = preg_replace('/[^a-zA-Z0-9_\-\.]/', '', $cookie_name);
        
        // Ensure it doesn't start with a number or special character
        if (preg_match('/^[0-9_\-\.]/', $cookie_name)) {
            $cookie_name = 'ref_' . $cookie_name;
        }
        
        // Limit length to 50 characters for practicality
        if (strlen($cookie_name) > 50) {
            $cookie_name = substr($cookie_name, 0, 50);
        }
        
        // If after sanitization it's empty or too short, use default
        if (strlen($cookie_name) < 3) {
            add_settings_error('trp_cookie_name', 'invalid_name', 
                __('Cookie name must be at least 3 characters long and contain valid characters.', 'tenant-referral-program'));
            return get_option('trp_cookie_name', TRP_COOKIE_NAME);
        }
        
        return $cookie_name;
    }
    
    /**
     * Sanitize agreement text input
     */
    public function sanitize_agreement_text($input) {
        // Allow basic HTML but sanitize for security
        $allowed_html = array(
            'strong' => array(),
            'em' => array(),
            'b' => array(),
            'i' => array(),
            'br' => array()
        );

        $sanitized = wp_kses($input, $allowed_html);

        // If empty, return default
        if (empty($sanitized)) {
            return 'I agree to the Terms & Conditions';
        }

        return $sanitized;
    }

    /**
     * Sanitize terms URL input
     */
    public function sanitize_terms_url($input) {
        // Clean the URL
        $url = esc_url_raw($input);

        // If empty, return default
        if (empty($url)) {
            return '/terms-and-conditions/';
        }

        // Allow both relative and absolute URLs
        $is_absolute = (bool) filter_var($url, FILTER_VALIDATE_URL);
        $is_relative = (is_string($url) && strlen($url) > 0 && $url[0] === '/'
            && preg_match('#^/[A-Za-z0-9\-._~/%?#=&+]*$#', $url));

        if (!$is_absolute && !$is_relative) {
            add_settings_error('trp_terms_url', 'invalid_terms_url',
                __('Please enter a valid URL or relative path for the Terms & Conditions page.', 'tenant-referral-program'));
            return get_option('trp_terms_url', '/terms-and-conditions/');
        }

        return $url;
    }

    /**
     * Sanitize API key input
     */
    public function sanitize_api_key($api_key) {
        $api_key = sanitize_text_field($api_key);
        
        // If empty, return empty (allow clearing the key)
        if (empty($api_key)) {
            return '';
        }
        
        // Check if validation bypass is enabled
        $bypass_validation = get_option('trp_bypass_api_validation', false);
        if ($bypass_validation) {
            add_settings_error('trp_api_key', 'bypass_enabled', 
                __('API validation bypassed - key saved without testing.', 'tenant-referral-program'), 'notice-info');
            return $api_key;
        }
        
        // Validate API key format
        if (!preg_match('/^trp_[a-f0-9]{32}$/', $api_key) && !preg_match('/^trp_[a-f0-9]{64}$/', $api_key)) {
            add_settings_error('trp_api_key', 'invalid_format', 
                __('Invalid API key format. Please check your API key.', 'tenant-referral-program'));
            return get_option('trp_api_key'); // Keep old value
        }
        
        // Test API key validity - but don't fail save if test fails
        $api_test_result = $this->test_api_key($api_key);
        if (!$api_test_result) {
            add_settings_error('trp_api_key', 'test_failed', 
                __('Warning: API key test failed. The key was saved but may not be working. Please check your base URL and network connection.', 'tenant-referral-program'), 'notice-warning');
        }
        
        return $api_key;
    }
    
    /**
     * Check rate limit for API testing
     */
    private function check_rate_limit($api_key) {
        $transient_key = 'trp_rate_limit_' . md5($api_key . get_current_user_id());
        $attempts = get_transient($transient_key) ?: 0;
        
        if ($attempts >= 10) { // 10 attempts per hour
            return false;
        }
        
        set_transient($transient_key, $attempts + 1, HOUR_IN_SECONDS);
        return true;
    }
    
    /**
     * Test API key validity
     */
    private function test_api_key($api_key) {
        // Check rate limit
        if (!$this->check_rate_limit($api_key)) {
            return false;
        }
        $base_url = get_option('trp_base_url', 'https://offertown.net');
        $test_url = rtrim($base_url, '/') . '/api/auth/test';
        
        // Determine SSL verification based on environment
        $ssl_verify = true;
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // Only disable SSL verification in debug mode for development
            $parsed_url = parse_url($test_url);
            if (in_array($parsed_url['host'], array('localhost', '127.0.0.1', '::1')) || 
                strpos($parsed_url['host'], '.local') !== false ||
                strpos($parsed_url['host'], 'ngrok') !== false) {
                $ssl_verify = false;
            }
        }
        
        $response = wp_remote_get($test_url, array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json',
                'ngrok-skip-browser-warning' => 'true'
            ),
            'timeout' => 10,
            'sslverify' => $ssl_verify
        ));
        
        if (is_wp_error($response)) {
            return false;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        return $status_code === 200;
    }
    
    /**
     * Render settings page
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Handle form submission messages
        if (isset($_GET['settings-updated'])) {
            add_settings_error(
                'trp_messages',
                'trp_message',
                __('Settings saved successfully.', 'tenant-referral-program'),
                'updated'
            );
        }
        
        settings_errors('trp_messages');
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <div class="trp-admin-header">
                <p><?php _e('Configure your tenant referral program settings below.', 'tenant-referral-program'); ?></p>
            </div>
            
            <form action="options.php" method="post">
                <?php
                settings_fields($this->settings_group);
                do_settings_sections($this->page_slug);
                submit_button(__('Save Settings', 'tenant-referral-program'));
                ?>
            </form>
            
            <div class="trp-admin-info">
                <h2><?php _e('Shortcodes', 'tenant-referral-program'); ?></h2>
                <p><?php _e('Use these shortcodes to display referral functionality on your site:', 'tenant-referral-program'); ?></p>
                
                <div class="trp-shortcode-info">
                    <h3><?php _e('Referral Widget', 'tenant-referral-program'); ?></h3>
                    <code>[tenant_referral_widget]</code>
                    <p><?php _e('Displays the referral program widget as a button that opens in a new tab.', 'tenant-referral-program'); ?></p>
                    
                    <h4><?php _e('Available Modes:', 'tenant-referral-program'); ?></h4>
                    <ul>
                        <li><code>mode="button"</code> - <?php _e('Button that opens in new tab (default, recommended)', 'tenant-referral-program'); ?></li>
                        <li><code>mode="modal"</code> - <?php _e('Button that opens in popup window', 'tenant-referral-program'); ?></li>
                        <li><code>mode="iframe"</code> - <?php _e('Embedded iframe (may require authentication)', 'tenant-referral-program'); ?></li>
                        <li><code>debug="yes"</code> - <?php _e('Debug mode to test different endpoints', 'tenant-referral-program'); ?></li>
                    </ul>
                </div>
                
                <div class="trp-shortcode-info">
                    <h3><?php _e('Signup Form', 'tenant-referral-program'); ?></h3>
                    <code>[tenant_referral_signup_form]</code>
                    <p><?php _e('Displays the referral signup form with tracking.', 'tenant-referral-program'); ?></p>
                    
                    <h4><?php _e('Available Attributes:', 'tenant-referral-program'); ?></h4>
                    <ul>
                        <li><code>show_title="yes"</code> - <?php _e('Show/hide the form title', 'tenant-referral-program'); ?></li>
                        <li><code>title="Custom Title"</code> - <?php _e('Custom form title', 'tenant-referral-program'); ?></li>
                        <li><code>success_redirect="/thank-you/"</code> - <?php _e('Redirect URL after successful signup', 'tenant-referral-program'); ?></li>
                    </ul>
                </div>
            </div>
            
            <div class="trp-admin-status">
                <h2><?php _e('System Status', 'tenant-referral-program'); ?></h2>
                <?php $this->render_system_status(); ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render general section
     */
    public function render_general_section() {
        echo '<p>' . __('Configure the basic settings for your referral program.', 'tenant-referral-program') . '</p>';
    }
    
    /**
     * Render tracking section
     */
    public function render_tracking_section() {
        echo '<p>' . __('Configure how referral codes are tracked and stored.', 'tenant-referral-program') . '</p>';
    }
    
    /**
     * Render API key field
     */
    public function render_api_key_field() {
        $value = get_option('trp_api_key', '');
        ?>
        <input type="password" 
               name="trp_api_key" 
               id="trp_api_key"
               value="<?php echo esc_attr($value); ?>" 
               class="regular-text" 
               placeholder="trp_abc123...">
        <p class="description">
            <?php _e('Get your API key from your referral program dashboard.', 'tenant-referral-program'); ?>
            <button type="button" id="toggle-api-key" class="button button-secondary">
                <?php _e('Show', 'tenant-referral-program'); ?>
            </button>
            <button type="button" id="test-api-key" class="button button-secondary">
                <?php _e('Test Connection', 'tenant-referral-program'); ?>
            </button>
        </p>
        <div id="api-key-status"></div>
        <?php
    }
    
    /**
     * Render base URL field
     */
    public function render_base_url_field() {
        $value = get_option('trp_base_url', 'https://offertown.net');
        ?>
        <input type="url" 
               name="trp_base_url" 
               value="<?php echo esc_attr($value); ?>" 
               class="regular-text" 
               required>
        <p class="description">
            <?php _e('The base URL of your referral program application (e.g., https://offertown.net)', 'tenant-referral-program'); ?>
        </p>
        <?php
    }
    
    /**
     * Render cookie expiry field
     */
    public function render_cookie_expiry_field() {
        $value = get_option('trp_cookie_expiry_days', TRP_COOKIE_EXPIRY_DAYS);
        ?>
        <input type="number" 
               name="trp_cookie_expiry_days" 
               value="<?php echo esc_attr($value); ?>" 
               min="1" 
               max="365" 
               class="small-text">
        <p class="description">
            <?php _e('How many days referral codes should be stored in cookies (1-365 days)', 'tenant-referral-program'); ?>
        </p>
        <?php
    }
    
    /**
     * Render cookie name field
     */
    public function render_cookie_name_field() {
        $value = get_option('trp_cookie_name', TRP_COOKIE_NAME);
        ?>
        <input type="text" 
               name="trp_cookie_name" 
               value="<?php echo esc_attr($value); ?>" 
               class="regular-text" 
               placeholder="e.g., MyBrand.tracking" 
               required>
        <p class="description">
            <?php _e('The name of the cookie used to store referral codes. Use your brand name for better identification and to avoid conflicts with other plugins.', 'tenant-referral-program'); ?><br>
            <strong><?php _e('Examples:', 'tenant-referral-program'); ?></strong> CANNA_referral, MyBrand_ref, CompanyName.tracking<br>
            <em><?php _e('Allowed characters: letters, numbers, underscores, dashes, and dots. Must be at least 3 characters.', 'tenant-referral-program'); ?></em>
        </p>
        <?php
    }
    
    /**
     * Render agreement text field
     */
    public function render_agreement_text_field() {
        $value = get_option('trp_agreement_text', 'I agree to the Terms & Conditions');
        ?>
        <textarea name="trp_agreement_text"
                  class="large-text"
                  rows="2"
                  placeholder="I agree to the Terms & Conditions"><?php echo esc_textarea($value); ?></textarea>
        <p class="description">
            <?php _e('The text displayed for the agreement checkbox on the signup form. Basic HTML allowed (strong, em, b, i, br).', 'tenant-referral-program'); ?>
        </p>
        <?php
    }

    /**
     * Render terms URL field
     */
    public function render_terms_url_field() {
        $value = get_option('trp_terms_url', '/terms-and-conditions/');
        ?>
        <input type="text"
               name="trp_terms_url"
               value="<?php echo esc_attr($value); ?>"
               class="regular-text"
               placeholder="/terms-and-conditions/">
        <p class="description">
            <?php _e('The URL where users will be redirected to read the Terms & Conditions. Can be a relative path (e.g., /terms/) or full URL (e.g., https://example.com/terms/).', 'tenant-referral-program'); ?>
        </p>
        <?php
    }

    /**
     * Render bypass validation field
     */
    public function render_bypass_validation_field() {
        $value = get_option('trp_bypass_api_validation', false);

        // Only show this field in development mode
        if (!defined('WP_DEBUG') || !WP_DEBUG) {
            ?>
            <p class="description" style="color: #999;">
                <?php _e('This setting is only available in development mode (WP_DEBUG must be enabled).', 'tenant-referral-program'); ?>
            </p>
            <?php
            return;
        }
        ?>
        <input type="checkbox"
               name="trp_bypass_api_validation"
               value="1"
               <?php checked($value, true); ?>>
        <p class="description">
            <strong style="color: #d63638;"><?php _e('⚠️ DEVELOPMENT ONLY:', 'tenant-referral-program'); ?></strong>
            <?php _e('Bypass API validation for testing. This should NEVER be enabled in production.', 'tenant-referral-program'); ?>
        </p>
        <?php
    }
    
    /**
     * Render development mode field
     */
    public function render_development_mode_field() {
        $value = get_option('trp_development_mode', false);
        ?>
        <input type="checkbox" 
               name="trp_development_mode" 
               value="1" 
               <?php checked($value, true); ?>>
        <p class="description">
            <?php _e('Enable development mode (cache-busting, verbose logging). Disable for production.', 'tenant-referral-program'); ?>
        </p>
        <?php
    }
    
    /**
     * Render system status
     */
    private function render_system_status() {
        $api_client = new Tenant_Referral_API_Client();
        $base_url = get_option('trp_base_url');
        $widget_url = $api_client->get_widget_url();
        $api_url = $api_client->get_track_api_url();
        
        ?>
        <table class="widefat">
            <tbody>
                <tr>
                    <td><strong><?php _e('Base URL:', 'tenant-referral-program'); ?></strong></td>
                    <td><?php echo esc_html($base_url); ?></td>
                </tr>
                <tr>
                    <td><strong><?php _e('Widget URL:', 'tenant-referral-program'); ?></strong></td>
                    <td><?php echo esc_html($widget_url); ?></td>
                </tr>
                <tr>
                    <td><strong><?php _e('API URL:', 'tenant-referral-program'); ?></strong></td>
                    <td><?php echo esc_html($api_url); ?></td>
                </tr>
                <tr>
                    <td><strong><?php _e('Cookie Name:', 'tenant-referral-program'); ?></strong></td>
                    <td><?php echo esc_html(get_option('trp_cookie_name', TRP_COOKIE_NAME)); ?></td>
                </tr>
                <tr>
                    <td><strong><?php _e('Cookie Expiry:', 'tenant-referral-program'); ?></strong></td>
                    <td><?php echo esc_html(get_option('trp_cookie_expiry_days', TRP_COOKIE_EXPIRY_DAYS)); ?> <?php _e('days', 'tenant-referral-program'); ?></td>
                </tr>
            </tbody>
        </table>
        <?php
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        if ('settings_page_' . $this->page_slug !== $hook) {
            return;
        }
        
        // Enqueue admin script
        wp_enqueue_script(
            'trp-admin-settings',
            plugin_dir_url(__FILE__) . 'js/admin-settings.js',
            array('jquery'),
            '1.0.0',
            true
        );
        
        // Localize script for AJAX
        wp_localize_script('trp-admin-settings', 'trp_admin_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('trp_test_api_key')
        ));
        
        // Add some basic admin styling
        // Use a dedicated inline style handle to guarantee output
        if (!wp_style_is('trp-admin-inline', 'enqueued')) {
            wp_register_style('trp-admin-inline', false);
            wp_enqueue_style('trp-admin-inline');
        }
        wp_add_inline_style('trp-admin-inline', '
            .trp-admin-header { margin-bottom: 20px; }
            .trp-admin-info { margin-top: 30px; padding: 20px; background: #f9f9f9; border-left: 4px solid #0073aa; }
            .trp-shortcode-info { margin-bottom: 20px; }
            .trp-shortcode-info code { background: #f1f1f1; padding: 2px 6px; border-radius: 3px; }
            .trp-admin-status { margin-top: 30px; }
            .trp-admin-status table { margin-top: 10px; }
            #api-key-status { margin-top: 10px; }
            #api-key-status.success { color: #46b450; }
            #api-key-status.error { color: #dc3232; }
            #api-key-status.testing { color: #0073aa; }
        ');
    }
    
    /**
     * AJAX handler for testing API key
     */
    public function ajax_test_api_key() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'trp_test_api_key')) {
            wp_die(__('Security check failed.', 'tenant-referral-program'));
        }
        
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die(__('Insufficient permissions.', 'tenant-referral-program'));
        }
        
        $api_key = sanitize_text_field($_POST['api_key']);
        $base_url = esc_url_raw($_POST['base_url']);
        
        if (empty($api_key) || empty($base_url)) {
            wp_send_json_error(__('API key and base URL are required.', 'tenant-referral-program'));
        }
        
        $test_url = rtrim($base_url, '/') . '/api/auth/test';
        
        $response = wp_remote_get($test_url, array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json',
                'ngrok-skip-browser-warning' => 'true'
            ),
            'timeout' => 10
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error($response->get_error_message());
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        if ($status_code === 200) {
            $data = json_decode($body, true);
            wp_send_json_success(array(
                'tenant_name' => isset($data['tenant']['name']) ? $data['tenant']['name'] : null
            ));
        } else {
            wp_send_json_error(sprintf(__('HTTP %d: %s', 'tenant-referral-program'), $status_code, $body));
        }
    }
}  