/**
 * Tenant Referral Program - Signup Form JavaScript
 * Extracted and adapted from template-referral-signup-test.php
 */

(function($) {
    'use strict';
    
    // Global object for public API
    window.TenantReferralSignupForm = {};
    
    // Cookie helper functions - exact logic from template
    function setCookie(name, value, days) {
        let expires = "";
        if (days) {
            const date = new Date();
            date.setTime(date.getTime() + (days * 24 * 60 * 60 * 1000));
            expires = "; expires=" + date.toUTCString();
        }
        const encodedValue = encodeURIComponent(value || "");
        let cookieString = name + "=" + encodedValue + expires + "; path=/; SameSite=Lax";
        if (window.location.protocol === 'https:') { 
            cookieString += "; Secure"; 
        }
        document.cookie = cookieString;
        console.log('TRP Signup: Cookie set:', name, '(encoded value: ' + encodedValue + ')', 'Expires in', days, 'days.');
    }

    function getCookie(name) {
        const nameEQ = name + "=";
        const ca = document.cookie.split(';');
        for (let i = 0; i < ca.length; i++) {
            let c = ca[i];
            while (c.charAt(0) === ' ') { 
                c = c.substring(1, c.length); 
            }
            if (c.indexOf(nameEQ) === 0) {
                const encodedValue = c.substring(nameEQ.length, c.length);
                try {
                    const decodedValue = decodeURIComponent(encodedValue);
                    console.log('TRP Signup: Cookie found:', name, '(decoded value: ' + decodedValue + ')');
                    return decodedValue;
                } catch (e) {
                    console.error('TRP Signup: Error decoding cookie:', name, encodedValue, e);
                    return encodedValue; 
                }
            }
        }
        console.log('TRP Signup: Cookie not found:', name);
        return null;
    }
    
    // Main initialization
    $(document).ready(function() {
        // Check for emergency override flag
        if (window.TRP_DISABLE_AJAX) {
            console.log('TRP Signup: AJAX disabled by override flag, skipping initialization');
            return;
        }
        
        // Check if this is a specific test form - if so, skip all initialization
        if (document.getElementById('inalan-signup-form-test') || 
            document.querySelector('form[action*="test-referral-signup"]') ||
            document.querySelector('.test-form-only')) {
            console.log('TRP Signup: Detected test form, skipping plugin initialization');
            return;
        }
        
        // Check if TRP_SignupConfig is available
        if (typeof TRP_SignupConfig === 'undefined') {
            console.error('TRP Signup: Configuration not loaded');
            return;
        }
        
        console.log('TRP Signup: Initializing with config:', TRP_SignupConfig);
        
        // Initialize referral tracking for each form
        $('.trp-signup-form').each(function() {
            initializeSignupForm($(this));
        });
        
        // Also check for forms that might be targeted by other selectors
        console.log('TRP Signup: Found .trp-signup-form elements:', $('.trp-signup-form').length);
        console.log('TRP Signup: All form elements on page:', $('form').length);
    });
    
    function initializeSignupForm($form) {
        const referralCodeField = $form.find('#trp-referral-code-capture');
        const referralAppliedNotice = $form.find('#trp-referral-code-applied-notice');
        const manualCodeArea = $form.find('#trp-manual-referral-code-area');
        const manualCodeInput = $form.find('#trp-manual-ref-code-input');
        const applyManualCodeButton = $form.find('#trp-apply-manual-ref-code-button');
        const manualCodeFeedback = $form.find('#trp-manual-ref-code-feedback');
        
        // Initialize referral code tracking
        initializeReferralTracking($form, referralCodeField, referralAppliedNotice, manualCodeArea);
        
        // Setup manual referral code entry
        setupManualCodeEntry(manualCodeInput, applyManualCodeButton, manualCodeFeedback, referralCodeField, referralAppliedNotice, manualCodeArea);
        
        // Setup AJAX form submission
        setupFormSubmission($form);
    }
    
    function initializeReferralTracking($form, referralCodeField, referralAppliedNotice, manualCodeArea) {
        const cookieName = TRP_SignupConfig.cookieName || 'tenant_ref_code';
        const cookieExpirationDays = TRP_SignupConfig.cookieExpiry || 30;
        let finalRefCode = null;
        
        try {
            const urlParams = new URLSearchParams(window.location.search);
            const refCodeFromUrl = urlParams.get('ref_code');

            if (refCodeFromUrl) {
                console.log('TRP Signup: ref_code FOUND in URL:', refCodeFromUrl);
                setCookie(cookieName, refCodeFromUrl, cookieExpirationDays);
                finalRefCode = refCodeFromUrl;

                // Track landing page visit
                if (TRP_SignupConfig.trackApiUrl) {
                    trackLandingPageVisit(refCodeFromUrl);
                }
            } else {
                const refCodeFromCookie = getCookie(cookieName);
                if (refCodeFromCookie) {
                    console.log('TRP Signup: ref_code NOT in URL, but FOUND in cookie:', refCodeFromCookie);
                    finalRefCode = refCodeFromCookie;
                } else {
                    console.log('TRP Signup: No ref_code found in URL or cookie.');
                }
            }

            // Update UI based on referral code status
            if (finalRefCode && referralCodeField.length) {
                referralCodeField.val(finalRefCode);
            }
            
            if (finalRefCode && referralAppliedNotice.length) {
                updateMainNotice(referralAppliedNotice, 'Referral code <strong>' + escapeHtml(finalRefCode) + '</strong> will be applied.', 'success');
                if (manualCodeArea.length) {
                    manualCodeArea.hide();
                }
            } else if (referralAppliedNotice.length) {
                updateMainNotice(referralAppliedNotice, 'No referral code active.', 'info');
                if (manualCodeArea.length) {
                    manualCodeArea.show();
                }
            }

        } catch (e) {
            console.error("TRP Signup: Error processing URL params or cookies for referral code:", e);
            if (referralAppliedNotice.length) {
                updateMainNotice(referralAppliedNotice, 'Error processing referral information.', 'error');
            }
            if (manualCodeArea.length) {
                manualCodeArea.show();
            }
        }
    }
    
    function trackLandingPageVisit(referralCode) {
        const payload = { 
            referralCode: referralCode, 
            event_type: 'LANDING_PAGE_VISIT' 
        };
        
        const requestOptions = {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(payload)
        };
        
        // Add ngrok headers if needed
        if (TRP_SignupConfig.trackApiUrl.includes('ngrok-free.app') || TRP_SignupConfig.trackApiUrl.includes('ngrok.io')) {
            requestOptions.headers['ngrok-skip-browser-warning'] = 'true';
        }
        
        fetch(TRP_SignupConfig.trackApiUrl, requestOptions)
            .then(response => {
                if (!response.ok) {
                    return response.json().catch(() => ({error: `Server responded with ${response.status}`})).then(body => {
                        throw new Error(`HTTP error ${response.status}: ${body.error || response.statusText}`);
                    });
                }
                return response.json();
            })
            .then(data => console.log('TRP Signup: LANDING_PAGE_VISIT tracked successfully.', data))
            .catch(error => console.error('TRP Signup: Error tracking LANDING_PAGE_VISIT:', error));
    }
    
    function setupManualCodeEntry(manualCodeInput, applyManualCodeButton, manualCodeFeedback, referralCodeField, referralAppliedNotice, manualCodeArea) {
        if (!applyManualCodeButton.length || !manualCodeInput.length || !manualCodeFeedback.length || !referralCodeField.length || !referralAppliedNotice.length) {
            return;
        }
        
        const cookieName = TRP_SignupConfig.cookieName || 'tenant_ref_code';
        const cookieExpirationDays = TRP_SignupConfig.cookieExpiry || 30;
        
        applyManualCodeButton.on('click', function() {
            const manualCode = manualCodeInput.val().trim();
            manualCodeFeedback.text(''); 

            if (manualCode) {
                console.log('TRP Signup: Applying manually entered code:', manualCode);
                setCookie(cookieName, manualCode, cookieExpirationDays);
                referralCodeField.val(manualCode);
                updateMainNotice(referralAppliedNotice, 'Referral code <strong>' + escapeHtml(manualCode) + '</strong> applied manually.', 'success');
                manualCodeInput.val(''); 
                manualCodeFeedback.text('Code applied!').css('color', 'var(--trp-success-text)');
            } else {
                console.log('TRP Signup: Manual code input was empty.');
                manualCodeFeedback.text('Please enter a code.').css('color', 'var(--trp-error-text)');
            }
        });
        
        // Allow Enter key to apply code
        manualCodeInput.on('keypress', function(e) {
            if (e.which === 13) {
                e.preventDefault();
                applyManualCodeButton.click();
            }
        });
    }
    
    function setupFormSubmission($form) {
        // Skip forms marked for traditional submission
        if ($form.hasClass('traditional-form-submission')) {
            console.log('TRP Signup: Skipping AJAX setup for traditional form');
            return;
        }
        
        $form.on('submit', function(e) {
            e.preventDefault();
            
            console.log('TRP Signup: Form submission started');
            
            // Add loading state
            $form.addClass('trp-loading');
            
            // Clear previous messages
            $form.find('.trp-signup-feedback').remove();
            
            // Prepare form data
            const formData = new FormData(this);
            formData.append('action', 'trp_process_signup');
            
            // Submit via AJAX
            $.ajax({
                url: TRP_SignupConfig.ajaxUrl,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                timeout: 30000,
                success: function(response) {
                    console.log('TRP Signup: AJAX response:', response);
                    handleFormResponse($form, response);
                },
                error: function(xhr, status, error) {
                    console.error('TRP Signup: AJAX error:', status, error);
                    handleFormError($form, 'Network error: ' + error);
                },
                complete: function() {
                    // Remove loading state
                    $form.removeClass('trp-loading');
                }
            });
        });
    }
    
    function handleFormResponse($form, response) {
        if (response.success) {
            // Success response
            const message = response.data.message || 'Registration successful!';
            showFormMessage($form, message, 'success');
            
            // Reset form
            $form[0].reset();
            
            // Clear referral code notice
            const referralAppliedNotice = $form.find('#trp-referral-code-applied-notice');
            if (referralAppliedNotice.length) {
                updateMainNotice(referralAppliedNotice, 'No referral code active.', 'info');
            }
            
            // Show manual code area again
            const manualCodeArea = $form.find('#trp-manual-referral-code-area');
            if (manualCodeArea.length) {
                manualCodeArea.show();
            }
            
            console.log('TRP Signup: Registration successful, user ID:', response.data.user_id);
            
        } else {
            // Error response
            const messages = response.data.messages || ['Registration failed'];
            showFormMessages($form, messages, 'error');
        }
    }
    
    function handleFormError($form, errorMessage) {
        showFormMessage($form, errorMessage, 'error');
    }
    
    function showFormMessage($form, message, type) {
        showFormMessages($form, [message], type);
    }
    
    function showFormMessages($form, messages, type) {
        let feedbackHtml = '<div class="trp-signup-feedback">';
        messages.forEach(function(message) {
            feedbackHtml += '<p class="trp-message-type-' + escapeHtml(type) + '">' + escapeHtml(message) + '</p>';
        });
        feedbackHtml += '</div>';
        
        // Insert at the top of the form
        $form.prepend(feedbackHtml);
        
        // Scroll to top of form
        $('html, body').animate({
            scrollTop: $form.offset().top - 50
        }, 500);
    }
    
    function updateMainNotice(noticeElement, message, type) {
        if (!noticeElement.length) return;
        
        noticeElement.html(message);
        const noticeStyle = noticeElement[0].style;
        
        switch(type) {
            case 'success':
                noticeStyle.color = 'var(--trp-success-text)';
                noticeStyle.padding = '10px';
                noticeStyle.backgroundColor = 'var(--trp-success-bg)';
                noticeStyle.border = '1px solid var(--trp-success-border)';
                noticeStyle.borderRadius = '4px';
                break;
            case 'info':
                noticeStyle.color = 'var(--trp-info-text)';
                noticeStyle.padding = '0';
                noticeStyle.backgroundColor = 'transparent';
                noticeStyle.border = 'none';
                break;
            default: 
                noticeStyle.color = 'var(--trp-error-text)';
                noticeStyle.padding = '10px';
                noticeStyle.backgroundColor = 'var(--trp-error-bg)';
                noticeStyle.border = '1px solid var(--trp-error-border)';
                noticeStyle.borderRadius = '4px';
        }
    }
    
    function escapeHtml(text) {
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.replace(/[&<>"']/g, function(m) { return map[m]; });
    }
    
    // Public API
    window.TenantReferralSignupForm = {
        setCookie: setCookie,
        getCookie: getCookie,
        trackLandingPageVisit: trackLandingPageVisit,
        escapeHtml: escapeHtml
    };
    
})(jQuery); 