<?php
/**
 * Message Manager for Tenant Referral Program
 * 
 * Handles message persistence across requests using WordPress transients
 * Replaces the $signup_messages array from template with cross-request storage
 */

if (!defined('ABSPATH')) {
    exit;
}

class Tenant_Referral_Message_Manager {
    
    /**
     * Transient key for storing messages
     */
    private $transient_key;
    
    /**
     * Constructor
     * 
     * @param string $unique_id Optional unique identifier for message storage
     */
    public function __construct($unique_id = null) {
        $this->transient_key = 'trp_messages_' . ($unique_id ?: $this->generate_session_id());
    }
    
    /**
     * Add a message to storage
     * 
     * @param string $type Message type (error, success, warning, info)
     * @param string $text Message text
     */
    public function add_message($type, $text) {
        $messages = $this->get_messages();
        $messages[] = array(
            'type' => $type,
            'text' => $text
        );
        
        // Store for 5 minutes (300 seconds)
        set_transient($this->transient_key, $messages, 300);
    }
    
    /**
     * Add multiple messages at once
     * 
     * @param array $messages Array of message arrays with 'type' and 'text' keys
     */
    public function add_messages($messages) {
        if (!is_array($messages)) {
            return;
        }
        
        $existing_messages = $this->get_messages();
        foreach ($messages as $message) {
            if (isset($message['type']) && isset($message['text'])) {
                $existing_messages[] = $message;
            }
        }
        
        set_transient($this->transient_key, $existing_messages, 300);
    }
    
    /**
     * Get all stored messages
     * 
     * @return array Array of messages
     */
    public function get_messages() {
        return get_transient($this->transient_key) ?: array();
    }
    
    /**
     * Check if there are any messages
     * 
     * @return bool True if messages exist
     */
    public function has_messages() {
        $messages = $this->get_messages();
        return !empty($messages);
    }
    
    /**
     * Get messages and clear them from storage
     * 
     * @return array Array of messages
     */
    public function get_and_clear_messages() {
        $messages = $this->get_messages();
        $this->clear_messages();
        return $messages;
    }
    
    /**
     * Clear all messages from storage
     */
    public function clear_messages() {
        delete_transient($this->transient_key);
    }
    
    /**
     * Generate a unique session ID for message storage
     * 
     * @return string Unique session identifier
     */
    private function generate_session_id() {
        // Try to use PHP session if available
        if (session_status() === PHP_SESSION_NONE) {
            @session_start();
        }
        
        if (session_id()) {
            return session_id();
        }
        
        // Fallback to unique ID based on user and time
        $user_id = get_current_user_id();
        $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
        
        return 'trp_' . md5($user_id . $ip_address . $user_agent . time());
    }
    
    /**
     * Render messages as HTML
     * Uses same styling structure as template
     * 
     * @param bool $clear_after_render Whether to clear messages after rendering
     * @return string HTML output
     */
    public function render_messages($clear_after_render = true) {
        $messages = $clear_after_render ? $this->get_and_clear_messages() : $this->get_messages();
        
        if (empty($messages)) {
            return '';
        }
        
        $output = '<div class="trp-signup-feedback">';
        foreach ($messages as $message) {
            $output .= sprintf(
                '<p class="trp-message-type-%s">%s</p>',
                esc_attr($message['type']),
                esc_html($message['text'])
            );
        }
        $output .= '</div>';
        
        return $output;
    }
    
    /**
     * Add error message
     * 
     * @param string $text Error message text
     */
    public function add_error($text) {
        $this->add_message('error', $text);
    }
    
    /**
     * Add success message
     * 
     * @param string $text Success message text
     */
    public function add_success($text) {
        $this->add_message('success', $text);
    }
    
    /**
     * Add warning message
     * 
     * @param string $text Warning message text
     */
    public function add_warning($text) {
        $this->add_message('warning', $text);
    }
    
    /**
     * Add info message
     * 
     * @param string $text Info message text
     */
    public function add_info($text) {
        $this->add_message('info', $text);
    }
    
    /**
     * Convert error array to messages and add them
     * 
     * @param array $errors Array of error strings
     */
    public function add_errors($errors) {
        if (!is_array($errors)) {
            return;
        }
        
        foreach ($errors as $error) {
            $this->add_error($error);
        }
    }
} 