<?php
/**
 * Shortcodes for Tenant Referral Program
 * 
 * Handles all shortcode functionality
 */

if (!defined('ABSPATH')) {
    exit;
}

class Tenant_Referral_Shortcodes {
    
    /**
     * API client instance
     */
    private $api_client;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->api_client = new Tenant_Referral_API_Client();
        $this->init_shortcodes();
        $this->setup_ajax_handlers(); // Register AJAX handlers immediately
    }
    
    /**
     * Initialize shortcodes
     */
    private function init_shortcodes() {
        add_shortcode('tenant_referral_widget', array($this, 'render_widget_shortcode'));
        add_shortcode('tenant_referral_signup', array($this, 'render_signup_form_shortcode'));
        add_shortcode('tenant_referral_signup_form', array($this, 'render_signup_form_shortcode')); // Documented name
        add_shortcode('tenant_referral_test', array($this, 'render_test_shortcode'));
        
        // Keep backward compatibility with existing shortcode
        add_shortcode('inalan_referral_widget', array($this, 'render_legacy_widget_shortcode'));
    }
    
    /**
     * Render the referral widget shortcode
     */
    public function render_widget_shortcode($atts) {
        $atts = shortcode_atts(array(
            'mode' => 'button',
            'debug' => 'no',
            'tenant' => ''
        ), $atts, 'tenant_referral_widget');
        
        $base_url = get_option('trp_base_url', '');
        
        // Fallback to legacy option if new option is empty
        if (empty($base_url)) {
            $base_url = get_option('inalan_referral_widget_url', '');
        }
        
        if (empty($base_url)) {
            return '<p style="color: red;">Error: No referral program URL configured. Please check your settings.</p>';
        }
        
        $widget_url = rtrim($base_url, '/') . TRP_WIDGET_ENDPOINT;
        
        // Add tenant parameter if provided, or auto-detect from API key
        $tenant_id = $atts['tenant'];
        if (empty($tenant_id)) {
            $tenant_id = $this->get_tenant_id_from_api_key();
        }
        
        if (!empty($tenant_id)) {
            $widget_url = add_query_arg('tenant', $tenant_id, $widget_url);
        }
        
        // Debug mode
        if ($atts['debug'] === 'yes') {
            return $this->render_debug_mode($base_url, $widget_url);
        }
        
        // Render based on mode
        switch ($atts['mode']) {
            case 'iframe':
                return $this->render_iframe_mode($widget_url);
            case 'modal':
                return $this->render_modal_mode($widget_url);
            case 'button':
            default:
                return $this->render_button_mode($widget_url);
        }
    }
    
    /**
     * Render button mode (opens in new tab)
     */
    private function render_button_mode($widget_url) {
        return sprintf(
            '<div class="trp-widget-container">
                <a href="%s" target="_blank" rel="noopener noreferrer" class="trp-widget-button">
                    Join Referral Program
                </a>
            </div>',
            esc_url($widget_url)
        );
    }
    
    /**
     * Render modal mode (opens in popup)
     */
    private function render_modal_mode($widget_url) {
        $popup_script = sprintf(
            "window.open('%s', 'referral_popup', 'width=800,height=600,scrollbars=yes,resizable=yes'); return false;",
            esc_js($widget_url)
        );
        
        return sprintf(
            '<div class="trp-widget-container">
                <button onclick="%s" class="trp-modal-button">
                    Open Referral Program
                </button>
            </div>',
            $popup_script
        );
    }
    
    /**
     * Render iframe mode (embedded)
     */
    private function render_iframe_mode($widget_url) {
        return sprintf(
            '<div class="trp-widget-container">
                <h2>Referral Program</h2>
                <iframe id="trp-widget-iframe" src="%s" frameborder="0">
                    <p>Your browser does not support iframes. <a href="%s" target="_blank">Click here to access the referral program</a>.</p>
                </iframe>
            </div>',
            esc_url($widget_url),
            esc_url($widget_url)
        );
    }
    
    /**
     * Render debug mode
     */
    private function render_debug_mode($base_url, $widget_url) {
        $possible_urls = array(
            'Widget URL' => $widget_url,
            'Base URL' => $base_url,
            'Alternative /widget' => rtrim($base_url, '/') . '/widget',
            'Alternative /widget/' => rtrim($base_url, '/') . '/widget/',
            'Root widget' => rtrim($base_url, '/') . '/widget.html'
        );
        
        $output = '<div class="trp-debug-container">';
        $output .= '<h4>Debug Mode - Referral Widget URLs</h4>';
        $output .= '<p>Testing different possible widget endpoints:</p>';
        $output .= '<ul>';
        
        foreach ($possible_urls as $label => $url) {
            $output .= sprintf(
                '<li><strong>%s:</strong> <a href="%s" target="_blank">%s</a></li>',
                esc_html($label),
                esc_url($url),
                esc_html($url)
            );
        }
        
        $output .= '</ul>';
        $output .= '<p><em>Click the links above to test which URL works for your referral program.</em></p>';
        $output .= '</div>';
        
        return $output;
    }
    
    /**
     * Legacy widget shortcode for backward compatibility
     */
    public function render_legacy_widget_shortcode($atts) {
        // Just redirect to the new shortcode
        return $this->render_widget_shortcode($atts);
    }
    
    /**
     * Render signup form shortcode
     * Complete implementation for Phase 3
     */
    public function render_signup_form_shortcode($atts) {
        $atts = shortcode_atts(array(
            'show_title' => 'yes',
            'title' => 'Sign Up & Refer Friends',
            'success_redirect' => '',
        ), $atts, 'tenant_referral_signup');
        
        // Enqueue form assets
        $this->enqueue_signup_form_assets();
        
        // Generate form HTML
        return $this->generate_signup_form_html($atts);
    }
    
    /**
     * Enqueue signup form assets
     */
    private function enqueue_signup_form_assets() {
        // Enqueue CSS
        wp_enqueue_style(
            'trp-signup-form',
            TRP_PLUGIN_URL . 'assets/css/signup-form.css',
            array(),
            TRP_VERSION
        );
        
        // Enqueue JavaScript
        wp_enqueue_script(
            'trp-signup-form',
            TRP_PLUGIN_URL . 'assets/js/signup-form.js',
            array('jquery'),
            TRP_VERSION,
            true
        );
        
        // Localize script with configuration
        wp_localize_script('trp-signup-form', 'TRP_SignupConfig', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('trp_signup_form'),
            'trackApiUrl' => $this->get_track_api_url(),
            'cookieName' => TRP_COOKIE_NAME,
            'cookieExpiry' => TRP_COOKIE_EXPIRY_DAYS,
            'debug' => defined('WP_DEBUG') && WP_DEBUG
        ));
    }
    
    /**
     * Setup AJAX handlers for form processing
     */
    private function setup_ajax_handlers() {
        // Only add handlers once
        if (!has_action('wp_ajax_trp_process_signup')) {
            add_action('wp_ajax_trp_process_signup', array($this, 'ajax_process_signup'));
            add_action('wp_ajax_nopriv_trp_process_signup', array($this, 'ajax_process_signup'));
        }
    }
    
    /**
     * AJAX handler for signup form processing
     * Updated to use friend signup process instead of creating WordPress users
     */
    public function ajax_process_signup() {
        // Verify nonce - check both possible field names
        $nonce_value = isset($_POST['nonce']) ? $_POST['nonce'] : (isset($_POST['trp_signup_nonce']) ? $_POST['trp_signup_nonce'] : '');
        
        if (!wp_verify_nonce($nonce_value, 'trp_signup_form')) {
            wp_send_json_error(array(
                'messages' => array('Security check failed. Please try again.')
            ));
        }
        
        // Initialize form handler
        $form_handler = new Tenant_Referral_Form_Handler();
        
        // Validate form data
        $validation = $form_handler->validate_form_data($_POST);
        
        if (!$validation['is_valid']) {
            wp_send_json_error(array(
                'messages' => $validation['errors']
            ));
        }
        
        // Process friend signup (no WordPress account created)
        $signup_result = $form_handler->process_friend_signup($validation['sanitized_data']);
        
        if (!$signup_result['success']) {
            wp_send_json_error(array(
                'messages' => array('Signup failed: ' . $signup_result['message'])
            ));
        }
        
        $success_messages = array('Thank you for signing up! Your information has been recorded.');
        
        // Track referral if present
        if (!empty($validation['sanitized_data']['referral_code'])) {
            $referral_result = $form_handler->track_referral_conversion(
                $signup_result['friend_id'],
                $validation['sanitized_data']['referral_code'],
                $validation['sanitized_data']
            );
            
            if ($referral_result['success']) {
                $success_messages[] = 'Your referral has been tracked successfully!';
            } else {
                $success_messages[] = 'Note: ' . $referral_result['message'];
            }
        } else {
            $success_messages[] = 'No referral code was provided with this signup.';
        }
        
        // Return success response
        wp_send_json_success(array(
            'message' => implode(' ', $success_messages),
            'friend_id' => $signup_result['friend_id'],
            'referral_tracked' => isset($referral_result) ? $referral_result['success'] : false
        ));
    }
    
    /**
     * Generate signup form HTML
     */
    private function generate_signup_form_html($atts) {
        // Get custom field values from settings
        $agreement_text = get_option('trp_agreement_text', 'I agree to the Terms & Conditions');
        $terms_url = get_option('trp_terms_url', '/terms-and-conditions/');

        ob_start();
        ?>
        <div class="trp-signup-form-container">
            <?php if ($atts['show_title'] === 'yes'): ?>
                <h2 class="trp-form-title"><?php echo esc_html($atts['title']); ?></h2>
            <?php endif; ?>

            <form id="trp-signup-form" class="trp-signup-form" method="POST">
                <?php wp_nonce_field('trp_signup_form', 'trp_signup_nonce'); ?>

                <p>
                    <label for="trp_user_email">Email: <span class="required-star">*</span></label>
                    <input type="email" name="user_email" id="trp_user_email" required>
                </p>

                <p>
                    <label for="trp_display_name">Display Name:</label>
                    <input type="text" name="display_name" id="trp_display_name">
                </p>

                <div class="trp-form-field">
                    <label for="trp_phone_number">Phone Number: <span class="required-star">*</span></label>
                    <input type="tel" name="phone_number" id="trp_phone_number" required>
                </div>

                <div class="trp-form-field">
                    <label for="trp_post_code">Post Code: <span class="required-star">*</span></label>
                    <input type="text" name="post_code" id="trp_post_code" required>
                </div>

                <div class="trp-form-field">
                    <label for="trp_address">Address: <span class="required-star">*</span></label>
                    <input type="text" name="address" id="trp_address" required>
                </div>

                <div class="trp-form-field trp-checkbox-group">
                    <input type="checkbox" name="consent_terms" id="trp_consent_terms" value="yes" required>
                    <label for="trp_consent_terms">
                        <?php
                        // Display agreement text (sanitized without links)
                        echo wp_kses($agreement_text, array('strong' => array(), 'em' => array(), 'b' => array(), 'i' => array(), 'br' => array()));

                        // Add link to terms if URL is provided
                        if (!empty($terms_url)) {
                            echo ' <a href="' . esc_url($terms_url) . '" target="_blank" rel="noopener noreferrer">' . esc_html__('(View)', 'tenant-referral-program') . '</a>';
                        }
                        ?>
                        <span class="required-star">*</span>
                    </label>
                </div>

                <input type="hidden" name="referral_code_capture" id="trp-referral-code-capture" value="">
                
                <div id="trp-referral-code-applied-notice" style="margin-bottom:15px; font-style:italic; color: #555;"></div>

                <div id="trp-manual-referral-code-area" style="display:none; margin-bottom:15px; padding:10px; border:1px solid var(--trp-border-color); border-radius:4px; background-color: #f9f9f9;">
                    <label for="trp-manual-ref-code-input" style="display:block; margin-bottom:5px; font-weight:normal; color: #333;">Have a referral code? Enter it here:</label>
                    <div class="trp-manual-code-input-group">
                        <input type="text" id="trp-manual-ref-code-input" style="flex-grow:1; padding:8px; border:1px solid var(--trp-border-color); border-radius:4px 0 0 4px;" placeholder="Enter code">
                        <button type="button" id="trp-apply-manual-ref-code-button" style="padding:8px 12px; border:1px solid var(--trp-red); border-left:none; background-color:var(--trp-red); color:white; border-radius:0 4px 4px 0; cursor:pointer;">Apply</button>
                    </div>
                    <div id="trp-manual-ref-code-feedback" style="font-size:0.9em; margin-top:5px;"></div>
                </div>

                <p>
                    <button type="submit" name="trp_signup_submit" class="trp-submit-button">
                        Sign Up & Complete Referral
                    </button>
                </p>
            </form>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Get track API URL
     */
    private function get_track_api_url() {
        $default_api_base_url = 'http://localhost:3000';
        $api_base_url_from_option = get_option('trp_base_url', $default_api_base_url);
        return rtrim($api_base_url_from_option, '/') . '/api/referrals/track';
    }
    
    /**
     * Render test shortcode for Phase 2 verification
     */
    public function render_test_shortcode($atts) {
        $atts = shortcode_atts(array(
            'section' => 'all'
        ), $atts, 'tenant_referral_test');
        
        ob_start();
        ?>
        <div class="trp-test-container" style="max-width: 800px; margin: 20px auto; padding: 20px; background: #f9f9f9; border-radius: 8px;">
            <h2 style="color: var(--trp-red, #E60000); text-align: center;">🚀 Phase 2 Test Interface</h2>
            
            <div class="trp-test-section" style="background: white; padding: 20px; margin: 20px 0; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                <h3>1. Configuration Test</h3>
                <p>Testing if TRP_Config is properly loaded:</p>
                <div id="trp-config-output" style="background: #f8f9fa; border: 1px solid #dee2e6; border-radius: 4px; padding: 15px; margin: 10px 0; font-family: monospace; font-size: 0.9em;">Loading...</div>
                <button onclick="testTRPConfiguration()" style="margin: 5px; padding: 8px 16px; background: var(--trp-red, #E60000); color: white; border: none; border-radius: 4px; cursor: pointer;">Test Configuration</button>
            </div>

            <div class="trp-test-section" style="background: white; padding: 20px; margin: 20px 0; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                <h3>2. Cookie Management Test</h3>
                <p>Testing cookie functionality:</p>
                <div id="trp-cookie-output" style="background: #f8f9fa; border: 1px solid #dee2e6; border-radius: 4px; padding: 15px; margin: 10px 0; font-family: monospace; font-size: 0.9em;">Ready for tests...</div>
                <button onclick="testTRPCookies()" style="margin: 5px; padding: 8px 16px; background: var(--trp-red, #E60000); color: white; border: none; border-radius: 4px; cursor: pointer;">Test Cookies</button>
            </div>

            <div class="trp-test-section" style="background: white; padding: 20px; margin: 20px 0; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                <h3>3. Manual Referral Code Test</h3>
                <p>Testing manual referral code entry:</p>
                
                <!-- Referral code notice area -->
                <div id="referral-code-applied-notice" style="margin-bottom:15px; font-style:italic; color: #555;"></div>

                <!-- Manual referral code area -->
                <div id="manual-referral-code-area" style="display:none; margin-bottom:15px; padding:10px; border:1px solid var(--trp-border-color, #ddd); border-radius:4px; background-color: #f9f9f9;">
                    <label for="manual_ref_code_input" style="display:block; margin-bottom:5px; font-weight:normal; color: #333;">Have a referral code? Enter it here:</label>
                    <div style="display:flex;">
                        <input type="text" id="manual_ref_code_input" style="flex-grow:1; padding:8px; border:1px solid var(--trp-border-color, #ddd); border-radius:4px 0 0 4px;" placeholder="Enter code">
                        <button type="button" id="apply_manual_ref_code_button" style="padding:8px 12px; border:1px solid var(--trp-red, #E60000); border-left:none; background-color:var(--trp-red, #E60000); color:white; border-radius:0 4px 4px 0; cursor:pointer;">Apply</button>
                    </div>
                    <div id="manual_ref_code_feedback" style="font-size:0.9em; margin-top:5px;"></div>
                </div>

                <!-- Hidden field for referral code capture -->
                <input type="hidden" name="referral_code_capture" id="referral_code_capture" value="">
                
                <div id="trp-manual-output" style="background: #f8f9fa; border: 1px solid #dee2e6; border-radius: 4px; padding: 15px; margin: 10px 0; font-family: monospace; font-size: 0.9em;">Manual entry ready...</div>
                <button onclick="showTRPManualEntry()" style="margin: 5px; padding: 8px 16px; background: var(--trp-red, #E60000); color: white; border: none; border-radius: 4px; cursor: pointer;">Show Manual Entry</button>
                <button onclick="checkTRPReferralField()" style="margin: 5px; padding: 8px 16px; background: var(--trp-red, #E60000); color: white; border: none; border-radius: 4px; cursor: pointer;">Check Hidden Field</button>
            </div>

            <div class="trp-test-section" style="background: white; padding: 20px; margin: 20px 0; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                <h3>4. Public API Test</h3>
                <p>Testing the TenantReferralProgram JavaScript API:</p>
                <div id="trp-api-output" style="background: #f8f9fa; border: 1px solid #dee2e6; border-radius: 4px; padding: 15px; margin: 10px 0; font-family: monospace; font-size: 0.9em;">API ready...</div>
                <button onclick="testTRPPublicAPI()" style="margin: 5px; padding: 8px 16px; background: var(--trp-red, #E60000); color: white; border: none; border-radius: 4px; cursor: pointer;">Test Public API</button>
                <button onclick="setTRPTestCode()" style="margin: 5px; padding: 8px 16px; background: var(--trp-red, #E60000); color: white; border: none; border-radius: 4px; cursor: pointer;">Set Test Code</button>
            </div>
        </div>

        <script>
        function testTRPConfiguration() {
            const output = document.getElementById('trp-config-output');
            let result = 'Configuration Test Results:\n\n';
            
            if (typeof TRP_Config !== 'undefined') {
                result += '✅ TRP_Config is loaded\n';
                result += 'Cookie Name: ' + TRP_Config.cookieName + '\n';
                result += 'Cookie Expiry: ' + TRP_Config.cookieExpiry + ' days\n';
                result += 'Track API URL: ' + TRP_Config.trackApiUrl + '\n';
                result += 'Base URL: ' + TRP_Config.baseUrl + '\n';
                result += 'Debug Mode: ' + TRP_Config.debug + '\n';
                result += 'Version: ' + TRP_Config.version + '\n';
            } else {
                result += '❌ TRP_Config is not loaded\n';
            }
            
            // Check for TenantReferralProgram with more debugging
            console.log('Testing for TenantReferralProgram:', typeof TenantReferralProgram);
            console.log('Window object keys containing "Tenant":', Object.keys(window).filter(key => key.includes('Tenant')));
            
            if (typeof TenantReferralProgram !== 'undefined') {
                result += '✅ TenantReferralProgram API is available\n';
                result += 'API Methods: ' + Object.keys(TenantReferralProgram).join(', ') + '\n';
            } else {
                result += '❌ TenantReferralProgram API is not available\n';
                result += 'Checking window.TenantReferralProgram: ' + typeof window.TenantReferralProgram + '\n';
            }
            
            output.textContent = result;
        }

        function testTRPCookies() {
            const output = document.getElementById('trp-cookie-output');
            if (typeof TenantReferralProgram !== 'undefined') {
                TenantReferralProgram.setCookie('test_cookie', 'test_value_123', 1);
                const value = TenantReferralProgram.getCookie('test_cookie');
                output.textContent = 'Cookie set and retrieved: ' + (value || 'null');
            } else {
                output.textContent = 'TenantReferralProgram API not available';
            }
        }

        function showTRPManualEntry() {
            const area = document.getElementById('manual-referral-code-area');
            if (area) {
                area.style.display = 'block';
                document.getElementById('trp-manual-output').textContent = 'Manual entry area shown - try entering a code!';
            }
        }

        function checkTRPReferralField() {
            const field = document.getElementById('referral_code_capture');
            const output = document.getElementById('trp-manual-output');
            if (field) {
                output.textContent = 'Hidden referral code field value: "' + field.value + '"';
            } else {
                output.textContent = 'Hidden referral code field not found';
            }
        }

        function testTRPPublicAPI() {
            const output = document.getElementById('trp-api-output');
            let result = 'Public API Test Results:\n\n';
            
            if (typeof TenantReferralProgram !== 'undefined') {
                result += '✅ TenantReferralProgram object exists\n';
                result += 'Available methods:\n';
                result += '- getCurrentReferralCode: ' + typeof TenantReferralProgram.getCurrentReferralCode + '\n';
                result += '- setReferralCode: ' + typeof TenantReferralProgram.setReferralCode + '\n';
                result += '- trackLandingPageVisit: ' + typeof TenantReferralProgram.trackLandingPageVisit + '\n';
                result += '- getCookie: ' + typeof TenantReferralProgram.getCookie + '\n';
                result += '- setCookie: ' + typeof TenantReferralProgram.setCookie + '\n';
                
                const currentCode = TenantReferralProgram.getCurrentReferralCode();
                result += '\nCurrent referral code: ' + (currentCode || 'none');
            } else {
                result += '❌ TenantReferralProgram object not found\n';
            }
            
            output.textContent = result;
        }

        function setTRPTestCode() {
            const output = document.getElementById('trp-api-output');
            if (typeof TenantReferralProgram !== 'undefined') {
                TenantReferralProgram.setReferralCode('WORDPRESS_TEST_123');
                output.textContent = 'Set referral code to: WORDPRESS_TEST_123\nCheck the manual entry section to see the update!';
            } else {
                output.textContent = 'TenantReferralProgram API not available';
            }
        }

        // Auto-run configuration test on load with delay to ensure scripts are loaded
        jQuery(document).ready(function() {
            setTimeout(function() {
                console.log('Auto-running configuration test...');
                testTRPConfiguration();
            }, 2000); // Increased delay to 2 seconds
        });
        </script>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Get tenant ID from configured API key
     * Makes a call to the SaaS API to resolve the tenant ID
     * 
     * @return string|null Tenant ID or null if not found
     */
    private function get_tenant_id_from_api_key() {
        $api_key = get_option('trp_api_key', '');
        
        if (empty($api_key)) {
            trp_debug_log('No API key configured for tenant ID lookup', 'TRP');
            return null;
        }
        
        $base_url = get_option('trp_base_url', '');
        if (empty($base_url)) {
            trp_debug_log('No base URL configured for tenant ID lookup', 'TRP');
            return null;
        }
        
        $test_url = rtrim($base_url, '/') . '/api/auth/test';
        
        $args = array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json'
            ),
            'timeout' => 10,
        );
        
        $response = wp_remote_get($test_url, $args);
        
        if (is_wp_error($response)) {
            trp_debug_log('Failed to get tenant ID - ' . $response->get_error_message(), 'TRP');
            return null;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            trp_debug_log('Failed to get tenant ID - HTTP ' . $status_code, 'TRP');
            return null;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (isset($data['tenant']['id'])) {
            $tenant_id = sanitize_text_field( $data['tenant']['id'] );
            trp_debug_log('Successfully resolved tenant ID ending with: ' . substr($tenant_id, -4), 'TRP');
            return $tenant_id;
        }
        
        trp_debug_log('No tenant ID found in API response', 'TRP');
        return null;
    }
} 