<?php
/**
 * Plugin Name: Tenant Referral Program
 * Plugin URI: https://www.offertown.net
 * Description: A comprehensive referral program plugin that allows users to sign up and track referrals through widgets and signup forms.
 * Version: 1.1.0
 * Author: OfferTown
 * Author URI: https://www.offertown.net
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: tenant-referral-program
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.4
 * Requires PHP: 7.4
 * Network: false
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('TRP_VERSION', '1.1.0');
define('TRP_PLUGIN_FILE', __FILE__);
define('TRP_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('TRP_PLUGIN_URL', plugin_dir_url(__FILE__));
define('TRP_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Define referral tracking constants
define('TRP_COOKIE_NAME', 'tenant_ref_code');
define('TRP_COOKIE_EXPIRY_DAYS', 30);
define('TRP_WIDGET_ENDPOINT', '/widget/');
define('TRP_API_ENDPOINT', '/api/referrals/track');

/**
 * Main plugin class
 */
class Tenant_Referral_Program {
    
    /**
     * Single instance of the plugin
     */
    private static $instance = null;
    
    /**
     * Get single instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->init_hooks();
        $this->load_dependencies();
    }
    
    /**
     * Initialize WordPress hooks
     */
    private function init_hooks() {
        add_action('init', array($this, 'init'));
        add_action('plugins_loaded', array($this, 'load_textdomain'));
        
        // Activation and deactivation hooks
        register_activation_hook(TRP_PLUGIN_FILE, array($this, 'activate'));
        register_deactivation_hook(TRP_PLUGIN_FILE, array($this, 'deactivate'));
    }
    
    /**
     * Load plugin dependencies
     */
    private function load_dependencies() {
        // Load core classes
        require_once TRP_PLUGIN_DIR . 'includes/class-api-client.php';
        require_once TRP_PLUGIN_DIR . 'includes/class-form-handler.php';
        require_once TRP_PLUGIN_DIR . 'includes/class-message-manager.php';
        require_once TRP_PLUGIN_DIR . 'includes/functions.php';
        
        // Load admin classes
        if (is_admin()) {
            require_once TRP_PLUGIN_DIR . 'admin/class-admin-settings.php';
        }
        
        // Always load public classes (needed for shortcodes in admin)
        require_once TRP_PLUGIN_DIR . 'public/class-shortcodes.php';
        require_once TRP_PLUGIN_DIR . 'public/class-assets.php';
    }
    
    /**
     * Initialize plugin
     */
    public function init() {
        // Initialize admin settings
        if (is_admin()) {
            new Tenant_Referral_Admin_Settings();
        }
        
        // Always initialize shortcodes (needed for both frontend and admin)
        new Tenant_Referral_Shortcodes();
        
        // Initialize assets (needed for both frontend and admin when shortcodes are used)
        new Tenant_Referral_Assets('tenant-referral-program', TRP_VERSION);
    }
    
    /**
     * Load plugin textdomain
     */
    public function load_textdomain() {
        load_plugin_textdomain(
            'tenant-referral-program',
            false,
            dirname(TRP_PLUGIN_BASENAME) . '/languages'
        );
    }
    
    /**
     * Plugin activation
     */
    public function activate() {
        // Set default options
        $default_options = array(
            'trp_base_url' => 'https://offertown.net',
            'trp_cookie_expiry_days' => TRP_COOKIE_EXPIRY_DAYS,
            'trp_cookie_name' => TRP_COOKIE_NAME
        );
        
        foreach ($default_options as $option_name => $default_value) {
            if (false === get_option($option_name)) {
                add_option($option_name, $default_value);
            }
        }
        
        // Flush rewrite rules
        flush_rewrite_rules();
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Flush rewrite rules
        flush_rewrite_rules();
    }
}

/**
 * Initialize the plugin
 */
function tenant_referral_program() {
    return Tenant_Referral_Program::get_instance();
}

// Start the plugin
tenant_referral_program(); 